'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';
import {
  categories as initialCategories,
  addProduct,
  removeProduct,
  updateProduct,
  addAddonToProduct,
  removeAddonFromProduct,
  addCategory,
  removeCategory,
} from '@/lib/menu-data';
import type { Category, Product, Addon } from '@/lib/types';
import { Button } from '@/components/ui/button';
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { useToast } from '@/hooks/use-toast';
import { Trash2, PlusCircle, ArrowLeft, Plus, X, FolderPlus, Edit, Save, Ban, Image as ImageIcon } from 'lucide-react';
import Link from 'next/link';
import AdminNav from '@/components/admin-nav';
import { Separator } from '@/components/ui/separator';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog"
import Image from 'next/image';

// We've moved the existing products list to its own component to keep this file clean.
function ExistingProducts({
    categories,
    onProductRemove,
    onProductUpdate,
    onAddonAdd,
    onAddonRemove,
    onCategoryRemove,
}: {
    categories: Category[];
    onProductRemove: (categoryId: string, productId: string) => void;
    onProductUpdate: (categoryId: string, productId: string, data: Partial<Product>) => void;
    onAddonAdd: (e: React.FormEvent, productId: string, categoryId: string) => void;
    onAddonRemove: (categoryId: string, productId: string, addonId: string) => void;
    onCategoryRemove: (categoryId: string) => void;
}) {
  const [newAddons, setNewAddons] = useState<{ [productId: string]: { name: string, price: string } }>({});
  const [editingProduct, setEditingProduct] = useState<Product | null>(null);

  const handleEditClick = (product: Product) => {
    setEditingProduct(JSON.parse(JSON.stringify(product)));
  };

  const handleCancelEdit = () => {
    setEditingProduct(null);
  }

  const handleSaveEdit = (categoryId: string) => {
    if (!editingProduct) return;
    onProductUpdate(categoryId, editingProduct.id, {
        name: editingProduct.name,
        description: editingProduct.description,
        price: Number(editingProduct.price),
        imageUrl: editingProduct.imageUrl
    });
    setEditingProduct(null);
  }
  
  const handleAddonInputChange = (productId: string, e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setNewAddons(prev => ({
      ...prev,
      [productId]: {
        ...prev[productId],
        [name]: value
      }
    }));
  };

  const handleInternalAddonAdd = (e: React.FormEvent, productId: string, categoryId: string) => {
    onAddonAdd(e, productId, categoryId);
    setNewAddons(prev => ({ ...prev, [productId]: { name: '', price: '' }}));
  }

  const formatPrice = (price: number) => {
    return price.toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });
  };
  
  const isEditing = (productId: string) => editingProduct?.id === productId;

  return (
      <div className="space-y-6">
        <h2 className="text-2xl font-bold">Produtos Existentes</h2>
        {categories.map((category) => (
          <div key={category.id}>
             <div className="flex justify-between items-center mb-4 border-b pb-2">
                <h3 className="text-xl font-semibold">{category.name}</h3>
                 <AlertDialog>
                  <AlertDialogTrigger asChild>
                    <Button variant="ghost" size="sm" className="text-destructive hover:bg-destructive/10 hover:text-destructive">
                      <Trash2 className="h-4 w-4 mr-2" />
                      Remover Categoria
                    </Button>
                  </AlertDialogTrigger>
                  <AlertDialogContent>
                    <AlertDialogHeader>
                      <AlertDialogTitle>Você tem certeza?</AlertDialogTitle>
                      <AlertDialogDescription>
                        Essa ação não pode ser desfeita. Isso irá deletar permanentemente a categoria e todos os produtos dentro dela.
                      </AlertDialogDescription>
                    </AlertDialogHeader>
                    <AlertDialogFooter>
                      <AlertDialogCancel>Cancelar</AlertDialogCancel>
                      <AlertDialogAction onClick={() => onCategoryRemove(category.id)} className="bg-destructive hover:bg-destructive/90">Deletar</AlertDialogAction>
                    </AlertDialogFooter>
                  </AlertDialogContent>
                </AlertDialog>
            </div>
            {category.products.length > 0 ? (
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                {category.products.map((product) => (
                  <Card key={product.id} className="flex flex-col">
                    {isEditing(product.id) ? (
                        <div className="p-4 space-y-2">
                           <Label htmlFor={`imageUrl-${product.id}`} className="flex items-center gap-2 text-sm"><ImageIcon className="h-4 w-4" /> URL da Imagem</Label>
                           <Input 
                                id={`imageUrl-${product.id}`}
                                value={editingProduct?.imageUrl}
                                onChange={(e) => setEditingProduct(p => p ? {...p, imageUrl: e.target.value} : null)}
                                placeholder="https://.../imagem.png"
                           />
                        </div>
                    ) : (
                        product.imageUrl && (
                             <div className="relative w-full h-48">
                                <Image 
                                    src={product.imageUrl} 
                                    alt={product.name} 
                                    fill 
                                    className="object-cover"
                                />
                             </div>
                        )
                    )}
                    <CardHeader className="pb-4">
                      {isEditing(product.id) ? (
                        <Input 
                            value={editingProduct?.name} 
                            onChange={(e) => setEditingProduct(p => p ? {...p, name: e.target.value} : null)}
                            className="text-2xl font-semibold leading-none tracking-tight p-0 h-auto border-0 focus-visible:ring-0"
                        />
                      ) : (
                        <CardTitle>{product.name}</CardTitle>
                      )}
                      {isEditing(product.id) ? (
                        <Textarea 
                            value={editingProduct?.description}
                            onChange={(e) => setEditingProduct(p => p ? {...p, description: e.target.value} : null)}
                            className="text-sm text-muted-foreground p-0 border-0 focus-visible:ring-0"
                        />
                      ) : (
                        <CardDescription className="flex-grow min-h-[40px]">{product.description}</CardDescription>
                      )}
                    </CardHeader>
                    <CardContent className="flex-grow space-y-4">
                        {isEditing(product.id) ? (
                             <Input 
                                type="number"
                                value={editingProduct?.price}
                                onChange={(e) => setEditingProduct(p => p ? {...p, price: Number(e.target.value)} : null)}
                                className="font-bold text-lg p-0 border-0 focus-visible:ring-0 h-auto"
                            />
                        ) : (
                             <p className="font-bold text-lg">{formatPrice(product.price)}</p>
                        )}
                       
                        
                        <Separator />

                        <div className="space-y-2">
                            <h4 className="font-semibold text-sm">Adicionais</h4>
                            {product.addons && product.addons.length > 0 ? (
                                <ul className="space-y-1">
                                    {product.addons.map(addon => (
                                        <li key={addon.id} className="flex items-center justify-between text-sm">
                                            <span>{addon.name} ({formatPrice(addon.price)})</span>
                                            <Button variant="ghost" size="icon" className="h-6 w-6 text-muted-foreground" onClick={() => onAddonRemove(category.id, product.id, addon.id)}>
                                                <X className="h-3 w-3" />
                                            </Button>
                                        </li>
                                    ))}
                                </ul>
                            ) : (
                                <p className="text-xs text-muted-foreground">Nenhum adicional.</p>
                            )}
                        </div>

                        <form onSubmit={(e) => handleInternalAddonAdd(e, product.id, category.id)} className="space-y-2 pt-2">
                           <div className="flex gap-2">
                             <Input 
                                name="name" 
                                placeholder="Nome do adicional" 
                                className="h-8 text-xs" 
                                value={newAddons[product.id]?.name || ''}
                                onChange={(e) => handleAddonInputChange(product.id, e)}
                              />
                             <Input 
                                name="price" 
                                type="number" 
                                placeholder="Preço" 
                                className="h-8 text-xs w-24"
                                step="0.01"
                                value={newAddons[product.id]?.price || ''}
                                onChange={(e) => handleAddonInputChange(product.id, e)}
                              />
                           </div>
                            <Button type="submit" size="sm" variant="secondary" className="w-full">
                                <Plus className="h-4 w-4 mr-1"/> Adicionar Adicional
                            </Button>
                        </form>
                    </CardContent>
                    <CardFooter className="flex flex-col gap-2">
                       {isEditing(product.id) ? (
                         <>
                           <Button size="sm" className="w-full" onClick={() => handleSaveEdit(category.id)}>
                                <Save className="h-4 w-4 mr-2" /> Salvar
                           </Button>
                           <Button variant="ghost" size="sm" className="w-full" onClick={handleCancelEdit}>
                               <Ban className="h-4 w-4 mr-2" /> Cancelar
                           </Button>
                         </>
                       ) : (
                        <>
                          <Button variant="outline" size="sm" className="w-full" onClick={() => handleEditClick(product)}>
                              <Edit className="h-4 w-4 mr-2" /> Editar Produto
                          </Button>
                          <Button
                            variant="destructive"
                            size="sm"
                            className="w-full"
                            onClick={() => onProductRemove(category.id, product.id)}
                          >
                            <Trash2 className="h-4 w-4 mr-2" />
                            Remover Produto
                          </Button>
                        </>
                       )}
                    </CardFooter>
                  </Card>
                ))}
              </div>
            ): (
                <p className="text-sm text-muted-foreground">Nenhum produto nesta categoria.</p>
            )}
          </div>
        ))}
      </div>
  )
}


export default function AdminProductsPage() {
  const [categories, setCategories] = useState<Category[]>(initialCategories);
  const [newProduct, setNewProduct] = useState({
    name: '',
    description: '',
    price: '',
    categoryId: initialCategories[0]?.id || '',
    imageUrl: ''
  });
  const [newCategoryName, setNewCategoryName] = useState('');

  const router = useRouter();
  const { toast } = useToast();

  const handleInputChange = (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>
  ) => {
    const { name, value } = e.target;
    setNewProduct((prev) => ({ ...prev, [name]: value }));
  };

  const handleSelectChange = (value: string) => {
    setNewProduct((prev) => ({ ...prev, categoryId: value }));
  };

  const handleAddProduct = (e: React.FormEvent) => {
    e.preventDefault();
    if (
      !newProduct.name ||
      !newProduct.price ||
      !newProduct.categoryId
    ) {
      toast({
        variant: 'destructive',
        title: 'Erro',
        description: 'Por favor, preencha todos os campos obrigatórios.',
      });
      return;
    }

    const added = addProduct(newProduct.categoryId, {
      name: newProduct.name,
      description: newProduct.description,
      price: parseFloat(newProduct.price),
      imageUrl: newProduct.imageUrl
    });

    if (added) {
        setCategories(JSON.parse(JSON.stringify(initialCategories)));
        toast({
            title: 'Sucesso!',
            description: `Produto "${added.name}" adicionado.`,
        });
        setNewProduct({
            name: '',
            description: '',
            price: '',
            categoryId: initialCategories[0]?.id || '',
            imageUrl: ''
        });
        router.refresh();
    }
  };

  const handleRemoveProduct = (categoryId: string, productId: string) => {
    removeProduct(categoryId, productId);
    setCategories(JSON.parse(JSON.stringify(initialCategories)));
    toast({
      title: 'Produto Removido',
      description: 'O produto foi removido do cardápio.',
    });
    router.refresh();
  };

  const handleUpdateProduct = (categoryId: string, productId: string, data: Partial<Product>) => {
    updateProduct(categoryId, productId, data);
    setCategories(JSON.parse(JSON.stringify(initialCategories)));
    toast({
        title: "Produto Atualizado!",
        description: "As informações do produto foram salvas."
    });
    router.refresh();
  };

  const handleAddAddon = (e: React.FormEvent, productId: string, categoryId: string) => {
    e.preventDefault();
    const form = e.target as HTMLFormElement;
    const nameInput = form.elements.namedItem('name') as HTMLInputElement;
    const priceInput = form.elements.namedItem('price') as HTMLInputElement;
    
    if (!nameInput.value || !priceInput.value) {
        toast({
            variant: "destructive",
            title: "Erro",
            description: "Preencha o nome e o preço do adicional."
        });
        return;
    }

    addAddonToProduct(categoryId, productId, {
        name: nameInput.value,
        price: parseFloat(priceInput.value)
    });
    
    setCategories(JSON.parse(JSON.stringify(initialCategories)));
     toast({
        title: 'Adicional Adicionado',
        description: `"${nameInput.value}" foi adicionado ao produto.`,
    });
    router.refresh();
  }

  const handleRemoveAddon = (categoryId: string, productId: string, addonId: string) => {
    removeAddonFromProduct(categoryId, productId, addonId);
    setCategories(JSON.parse(JSON.stringify(initialCategories)));
     toast({
        title: 'Adicional Removido',
        description: 'O adicional foi removido do produto.',
    });
    router.refresh();
  }

  const handleAddCategory = (e: React.FormEvent) => {
    e.preventDefault();
    if (!newCategoryName.trim()) {
      toast({
        variant: 'destructive',
        title: 'Erro',
        description: 'O nome da categoria não pode estar vazio.',
      });
      return;
    }
    addCategory(newCategoryName.trim());
    setCategories(JSON.parse(JSON.stringify(initialCategories)));
    toast({
      title: 'Categoria Adicionada!',
      description: `A categoria "${newCategoryName.trim()}" foi criada.`,
    });
    setNewCategoryName('');
    router.refresh();
  };

  const handleRemoveCategory = (categoryId: string) => {
    removeCategory(categoryId);
    setCategories(JSON.parse(JSON.stringify(initialCategories)));
    toast({
      title: 'Categoria Removida',
      description: 'A categoria e todos os seus produtos foram removidos.',
    });
    setNewProduct((prev) => ({ ...prev, categoryId: initialCategories[0]?.id || '' }));
    router.refresh();
  };
  
  return (
    <div className="max-w-7xl mx-auto p-4 md:p-6">
      <div className="flex items-center gap-4 mb-6">
        <Link href="/">
            <Button variant="outline" size="icon">
                <ArrowLeft className="h-4 w-4" />
            </Button>
        </Link>
        <h1 className="text-3xl font-bold">Gerenciar Cardápio</h1>
      </div>

      <AdminNav />

      <Card className="mb-8">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <FolderPlus className="h-6 w-6" />
            Adicionar Nova Categoria
          </CardTitle>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleAddCategory} className="flex gap-2">
            <Input
              value={newCategoryName}
              onChange={(e) => setNewCategoryName(e.target.value)}
              placeholder="Ex: Combos"
              required
            />
            <Button type="submit">Adicionar Categoria</Button>
          </form>
        </CardContent>
      </Card>

      <Card className="mb-8">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <PlusCircle className="h-6 w-6" />
            Adicionar Novo Produto
          </CardTitle>
          <CardDescription>
            Preencha o formulário para adicionar um novo item ao cardápio.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleAddProduct} className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="name">Nome do Produto</Label>
                <Input
                  id="name"
                  name="name"
                  value={newProduct.name}
                  onChange={handleInputChange}
                  placeholder="Ex: Pizza de Calabresa"
                  required
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="price">Preço</Label>
                <Input
                  id="price"
                  name="price"
                  type="number"
                  step="0.01"
                  value={newProduct.price}
                  onChange={handleInputChange}
                  placeholder="Ex: 42.50"
                  required
                />
              </div>
            </div>
             <div className="space-y-2">
              <Label htmlFor="imageUrl">URL da Imagem</Label>
              <Input
                id="imageUrl"
                name="imageUrl"
                value={newProduct.imageUrl}
                onChange={handleInputChange}
                placeholder="https://.../imagem.png"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="description">Descrição</Label>
              <Textarea
                id="description"
                name="description"
                value={newProduct.description}
                onChange={handleInputChange}
                placeholder="Molho de tomate, mussarela, calabresa fatiada e cebola."
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="categoryId">Categoria</Label>
              <Select
                name="categoryId"
                onValueChange={handleSelectChange}
                value={newProduct.categoryId}
                required
              >
                <SelectTrigger id="categoryId">
                  <SelectValue placeholder="Selecione uma categoria" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map((cat) => (
                    <SelectItem key={cat.id} value={cat.id}>
                      {cat.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <Button type="submit">Adicionar Produto</Button>
          </form>
        </CardContent>
      </Card>
      
      <Separator className="my-8" />
      
      <ExistingProducts 
        categories={categories}
        onProductRemove={handleRemoveProduct}
        onProductUpdate={handleUpdateProduct}
        onAddonAdd={handleAddAddon}
        onAddonRemove={handleRemoveAddon}
        onCategoryRemove={handleRemoveCategory}
      />
    </div>
  );
}
